API_KEY = "OPENAI API KEY"  # Replace with your actual key

import os
import time
import requests
import argparse

"""
storyboard_generator.py

Batch-generate AI storyboard frames into the fixed 'images' folder with live progress.

Author: Thomas Sweet / Anthro Teacher (Cio)

Usage:
  python storyboard_generator.py [--prompts FILE] [--descriptions FILE] [--variations N]

Options:
  --prompts FILE      Path to prompts file (default: prompts.txt)
  --descriptions FILE Path to descriptions file (default: descriptions.txt)
  --variations N      Number of variations per prompt (default: 2)
  -h, --help          Show this help message and exit
"""

# ─── Argument parsing ───────────────────────────────────────────────────────────
parser = argparse.ArgumentParser(
    description="Batch-generate AI storyboard frames into the 'images' folder with live progress."
)
parser.add_argument(
    '--prompts', dest='prompts_file', default='prompts.txt',
    help='Path to prompts file (default: prompts.txt)'
)
parser.add_argument(
    '--descriptions', dest='descriptions_file', default='descriptions.txt',
    help='Path to descriptions file (default: descriptions.txt)'
)
parser.add_argument(
    '--variations', type=int, default=2,
    help='Number of variations per prompt (default: 2)'
)
args = parser.parse_args()

# ─── Configuration ─────────────────────────────────────────────────────────────
PROMPTS_FILE  = args.prompts_file
DESCRIPTIONS  = args.descriptions_file
OUTPUT_DIR    = "images"
MODEL         = "dall-e-3"
SIZE          = "1024x1024"
VARIATIONS    = args.variations
DELAY_SECONDS = 1.0

# ─── Prepare output directory ───────────────────────────────────────────────────
os.makedirs(OUTPUT_DIR, exist_ok=True)

# ─── Load descriptions ──────────────────────────────────────────────────────────
descs = {}
try:
    with open(DESCRIPTIONS, encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line or ":" not in line:
                continue
            _, rest = line.split(". ", 1)
            name, text = rest.split(":", 1)
            descs[name.strip()] = text.strip()
except FileNotFoundError:
    print(f"Error: Descriptions file '{DESCRIPTIONS}' not found.")
    exit(1)

# ─── Read all scene prompts ─────────────────────────────────────────────────────
prompts = []
try:
    with open(PROMPTS_FILE, encoding="utf-8") as f:
        for line in f:
            entry = line.strip()
            if entry:
                prompts.append(entry)
except FileNotFoundError:
    print(f"Error: Prompts file '{PROMPTS_FILE}' not found.")
    exit(1)

if not prompts:
    print(f"Error: No prompts found in '{PROMPTS_FILE}'")
    exit(1)

# ─── API setup ─────────────────────────────────────────────────────────────────
ENDPOINT = "https://api.openai.com/v1/images/generations"
HEADERS  = {"Authorization": f"Bearer {API_KEY}", "Content-Type": "application/json"}

# ─── Helpers ────────────────────────────────────────────────────────────────────
def format_time(seconds):
    m, s = divmod(int(seconds), 60)
    h, m = divmod(m, 60)
    return f"{h:02d}:{m:02d}:{s:02d}"

# ─── Progress tracking setup ────────────────────────────────────────────────────
total_images = len(prompts) * VARIATIONS
images_done = 0
start_time = time.time()

# ─── Epic scene keywords ───────────────────────────────────────────────────────
epic_keywords = ["battle", "storm", "clash", "war", "thunderstorm", "siege"]

# ─── Iterate through all prompts ────────────────────────────────────────────────
for idx, entry in enumerate(prompts, start=1):
    try:
        prefix, scene, quote = entry.split(":", 2)
        _, names_str = prefix.split(". ", 1)
        names = [n.strip() for n in names_str.split(",")]
        scene = scene.strip()
        quote = quote.strip()
    except ValueError:
        print(f"⚠️ Skipping invalid prompt at line {idx}: '{entry}'")
        continue

    lower = scene.lower()
    include_lettering = "kablu" in lower
    is_epic = any(k in lower for k in epic_keywords)

    # Construct the full prompt with descriptions
    parts = []
    if not include_lettering:
        parts.append("No text or lettering in the image.")
    for n in names:
        if n in descs:
            parts.append(f"{n}: {descs[n]}")
    if not is_epic:
        parts.append("Use soft, diffused lighting and a gentle color palette with a calm, serene mood.")
    parts.append(f"Scene Instructions: {scene}")
    parts.append(f"Quote Reference: {quote}")
    # Limit prompt length to ~2,000 characters
    full_prompt = " ".join(parts)
    if len(full_prompt) > 1800:
        full_prompt = full_prompt[:1800] + "..."

    print(f"\n--- Prompt {idx}/{len(prompts)}: {scene[:30]}... ---")

    for var in range(1, VARIATIONS + 1):
        print(f"🔄 Generating variation {var}/{VARIATIONS} for prompt {idx}...")
        payload = {"model": MODEL, "prompt": full_prompt, "n": 1, "size": SIZE, "response_format": "url"}
        t0 = time.time()
        resp = requests.post(ENDPOINT, headers=HEADERS, json=payload)
        if not resp.ok:
            print(f"❌ Error {resp.status_code} on prompt {idx}, var {var}: {resp.text}")
            break

        data = resp.json().get("data", [])
        if not data:
            print(f"⚠️ No data returned for prompt {idx}, var {var}")
            continue

        url = data[0].get("url")
        img_data = requests.get(url).content
        filename = f"scene{idx}-{var}.png"
        path = os.path.join(OUTPUT_DIR, filename)
        with open(path, "wb") as out:
            out.write(img_data)
        print(f"✅ Saved {path}")

        # Update progress
        images_done += 1
        elapsed = time.time() - start_time
        avg_per_image = elapsed / images_done
        remaining = total_images - images_done
        eta = avg_per_image * remaining
        print(f"⏱ Elapsed: {format_time(elapsed)} | ETA: {format_time(eta)}")

        time.sleep(DELAY_SECONDS)

# Final summary
print()
print(f"🎉 All done in {format_time(time.time() - start_time)}! Your images are in the '{OUTPUT_DIR}' folder.")